/***********************************************************************
 * \file: util_mpf.c
 * 
 * \version: $Revision: 1.13 $
 * 
 * \release: $Name: UTIL_r2_E05 $
 *
 * Implementation for fixed memory pools.
 * 
 * \component: Utility library
 * 
 * \author T. Polle / ADIT / SWG / tpolle(o)de.adit-jv.com
 * 
 * \copyright: (c) 2003 - 2008 ADIT Corporation 
 *
 ***********************************************************************/


#include <stdlib.h>
#include <string.h>

#include "util_lib.h"

#if defined(MEM_DEBUG)
#  define UTIL_MPF_GET_PTN     0x4D504642
#  define UTIL_MPF_REL_PTN     0x4246504D
#endif /* defined(MEM_DEBUG) */

typedef struct UTIL_mpf_block_t UTIL_mpf_block_t;

struct UTIL_mpf_block_t
{
    UTIL_mpf_block_t*  next;
};

struct UTIL_mpf_t
{
    U8*                mem;
    UTIL_mpf_block_t*  free;
    S32                freecnt;
    S32                mpfcnt;
    S32                blfsz;
    U8                 dsname[8];
};

/*
 * Initialise the memory blocks in the pool.
 */
EXPORT void UTIL_init_mpf(UTIL_mpf_t* pool)
{
    UTIL_mpf_block_t*  block  = (UTIL_mpf_block_t*)((VP)pool->mem);
    block->next = NULL;
    while ((U8*)(VP)block < (pool->mem + ((pool->mpfcnt - 1) * pool->blfsz)))
    {
        VP  previous  = block;
        block         = (UTIL_mpf_block_t*)((VP)((U8*)(VP)block + pool->blfsz));
        block->next   = previous;
    }
    pool->free = block;
}

/* PRQA: QAC Message 5118: Heap allocation necessary. */
/* PRQA S 5118 L1 */
EXPORT UTIL_mpf_t* UTIL_cre_mpf(UTIL_CMPF_t* pk_cmpf)
{
    UTIL_mpf_t* pool = NULL;

    pk_cmpf = pk_cmpf;

    if (((pk_cmpf->mpfcnt * pk_cmpf->blfsz) > 0) && (pk_cmpf->blfsz > 0))
    {
        /* Create a new pool only if the total size and block size is greater
           than 0.
        */
        if (UTIL_TA_BUF == (UTIL_TA_BUF & pk_cmpf->mpfatr))
        {
            pool = pk_cmpf->mem;
        }
        else
        {
            pool = malloc(sizeof(UTIL_mpf_t)
                          + (U32)(pk_cmpf->mpfcnt * pk_cmpf->blfsz));
        }
        if (NULL != pool)
        {
            pool->mem     = (U8*)(VP)pool + sizeof(UTIL_mpf_t);
            pool->free    = NULL;
            pool->freecnt = pk_cmpf->mpfcnt;
            pool->mpfcnt  = pk_cmpf->mpfcnt;
            pool->blfsz   = pk_cmpf->blfsz;
            if (0 != (pk_cmpf->mpfatr & TA_DSNAME))
            {
                strncpy((VP)pool->dsname, (VP)pk_cmpf->dsname,
                        sizeof(pool->dsname));
            }
            UTIL_init_mpf(pool);
        }
    }
    return pool;
}
  
EXPORT S32 UTIL_del_mpf(UTIL_mpf_t* pool)
{
    if (NULL != pool)
    {
        free(pool);
    }
    return E_OK;
}
/* PRQA L:L1 */
  
EXPORT S32 UTIL_get_mpf(UTIL_mpf_t* pool, VP* block, TMO tmout)
{
    S32 rc = E_OK;
    tmout = tmout;
    if (NULL != pool->free)
    {
        (*block)   = pool->free;
        pool->free = pool->free->next;
        pool->freecnt--;
#if defined(MEM_DEBUG)
        memset((*block), UTIL_MPF_GET_PTN, pool->blfsz);
#endif /* defined(MEM_DEBUG) */
    }
    else
    {
        (*block) = NULL;
        rc       = E_NOMEM;
    }
    return rc;
}

EXPORT S32 UTIL_rel_mpf(UTIL_mpf_t* pool, VP block_)
{
    if (NULL != block_)
    {
        UTIL_mpf_block_t* block = block_;
#if defined(MEM_DEBUG)
        memset(block_, UTIL_MPF_REL_PTN, pool->blfsz);
#endif /* defined(MEM_DEBUG) */
        block->next = pool->free;
        pool->free  = block;
        pool->freecnt++;
    }
    return E_OK;
}

EXPORT U32 UTIL_sizeof_cmpf(void)
{
    return (U32)sizeof(UTIL_mpf_t);
}

EXPORT S32 UTIL_blfsz_mpf(UTIL_mpf_t* pool)
{
    pool = pool;
    return pool->blfsz;
}
